#!/bin/bash
# SandGraph + MCP Quick Installation Script
#
# Conda installation steps:
# 1. Create new conda environment:
#    conda create -n sandgraph python=3.11
# 2. Activate environment:
#    conda activate sandgraph
# 3. Run this script:
#    ./quick_install.sh

# Check Python version
python_version=$(python3 -c 'import sys; print(".".join(map(str, sys.version_info[:2])))')
if (( $(echo "$python_version < 3.8" | python3 -c "import sys; print(float(sys.stdin.read()) < 3.8)") )); then
    echo "Error: Python 3.8 or higher required, current version: $python_version"
    exit 1
fi

echo " Starting SandGraph + Official MCP SDK installation..."

# 1. Install basic dependencies
echo " Installing basic dependencies..."
pip install numpy pandas scipy networkx matplotlib || { echo "Error: Basic dependencies installation failed"; exit 1; }

# 2. Install official MCP SDK
echo " Installing official MCP SDK..."
pip install 'mcp[cli]' || { echo "Error: MCP SDK installation failed"; exit 1; }

# 3. Install trading-related dependencies
echo " Installing trading-related dependencies..."
pip install backtrader==1.9.76.123 mplfinance==0.12.10b0 yfinance==0.2.36 alpaca-trade-api==3.0.2 || { echo "Error: Trading dependencies installation failed"; exit 1; }

# 4. Install other dependencies
echo " Installing other dependencies..."
pip install anthropic==0.3.0 colorama==0.4.6 || { echo "Error: Other dependencies installation failed"; exit 1; }

# 4.1 Install gymnasium (replacement for gym)
echo " Installing gymnasium..."
pip install gymnasium || { echo "Error: gymnasium installation failed"; exit 1; }

# 4.2 Install trading-gym
pip install trading-gym==0.1.8 || { echo "Error: trading-gym installation failed"; exit 1; }

# 5. Install SandGraph (development version)
echo " Installing SandGraph..."
pip install -e . || { echo "Error: SandGraph installation failed"; exit 1; }

# 6. Install PyTorch and related dependencies
echo " Installing PyTorch and related dependencies..."
pip install torch transformers accelerate tiktoken einops transformers_stream_generator || { echo "Error: PyTorch and related dependencies installation failed"; exit 1; }

# 7. Install AReaL framework and optimization dependencies
echo " Installing AReaL framework and optimization dependencies..."
pip install "numpy>=1.21.0" "torch>=1.9.0" "psutil>=5.8.0" || { echo "Error: Core optimization dependencies installation failed"; exit 1; }

# 8. Install visualization and monitoring dependencies
echo " Installing visualization and monitoring dependencies..."
pip install "matplotlib>=3.5.0" "seaborn>=0.11.0" "plotly>=5.0.0" "pandas>=1.3.0" || { echo "Error: Visualization dependencies installation failed"; exit 1; }

# 9. Install optional advanced dependencies (non-critical)
echo " Installing optional advanced dependencies..."
pip install "scikit-learn>=1.0.0" "redis>=4.0.0" "python-memcached>=1.59" || { echo "Warning: Some optional dependencies failed to install, continuing..."; }

# 10. Install development and testing tools
echo " Installing development and testing tools..."
pip install "pytest>=6.0.0" "black>=21.0.0" "flake8>=3.9.0" || { echo "Warning: Development tools installation failed, continuing..."; }

echo ""
echo " Installation completed!"
echo ""
echo " Run the following commands to verify installation:"
echo "python -c \"from mcp.server.fastmcp import FastMCP; print('MCP SDK installation successful')\""
echo "python -c \"from sandgraph import check_mcp_availability; print(check_mcp_availability())\""
echo ""
echo " Next steps:"
echo "1. Run demo: python demo/sandbox_optimization.py"
echo "2. Start MCP server: python demo/mcp_server_example.py"
echo "3. View documentation: cat README.md" 